import { loadUIExtensionSpecificiationsFromPlugins, loadFunctionSpecificationsFromPlugins, } from '../../private/plugins/extension.js';
import { platformAndArch } from '@shopify/cli-kit/node/os';
import { memoize } from '@shopify/cli-kit/common/function';
import { isShopify } from '@shopify/cli-kit/node/context/local';
import { joinPath, dirname } from '@shopify/cli-kit/node/path';
import { glob } from '@shopify/cli-kit/node/fs';
import { fileURLToPath } from 'url';
export async function loadUIExtensionSpecifications(config) {
    const local = await loadLocalUIExtensionsSpecifications();
    const plugins = await loadUIExtensionSpecificiationsFromPlugins(config);
    return [...local, ...plugins];
}
export async function loadFunctionSpecifications(config) {
    const local = await loadLocalFunctionSpecifications();
    const plugins = await loadFunctionSpecificationsFromPlugins(config);
    return [...local, ...plugins];
}
export async function loadLocalUIExtensionsSpecifications() {
    return memoizedLoadSpecs('ui-specifications');
}
export async function loadLocalFunctionSpecifications() {
    const isShopifyUser = await isShopify();
    return (await memoizedLoadSpecs('function-specifications')).filter((spec) => !spec.gated || isShopifyUser);
}
export async function loadThemeSpecifications() {
    return memoizedLoadSpecs('theme-specifications');
}
/**
 * Load all specifications from the local file system AND plugins
 */
export async function loadExtensionsSpecifications(config) {
    const ui = await loadUIExtensionSpecifications(config);
    const functions = await loadFunctionSpecifications(config);
    const theme = await loadThemeSpecifications();
    return [...ui, ...functions, ...theme];
}
/**
 * Load all specifications ONLY from the local file system
 */
export async function loadLocalExtensionsSpecifications() {
    const ui = await loadLocalUIExtensionsSpecifications();
    const functions = await loadLocalFunctionSpecifications();
    const theme = await loadThemeSpecifications();
    return [...ui, ...functions, ...theme];
}
const memoizedLoadSpecs = memoize(loadSpecifications);
async function loadSpecifications(directoryName) {
    /**
     * When running tests, "await import('.../spec..ts')" is handled by Vitest which does
     * transform the TS module into a JS one before loading it. Hence the inclusion of .ts
     * in the list of files.
     */
    const url = joinPath(dirname(fileURLToPath(import.meta.url)), joinPath(directoryName, '*.{js,ts}'));
    let files = await glob(url, { ignore: ['**.d.ts', '**.test.ts'] });
    // From Node 18, all windows paths must start with file://
    const { platform } = platformAndArch();
    if (platform === 'windows') {
        files = files.map((file) => `file://${file}`);
    }
    const promises = files.map((file) => import(file));
    const modules = await Promise.all(promises);
    const specs = modules.map((module) => module.default);
    return specs;
}
//# sourceMappingURL=specifications.js.map