import { BaseFunctionConfigurationSchema, ZodSchemaType } from './schemas.js';
import { ExtensionCategory, GenericSpecification, FunctionExtension } from '../app/extensions.js';
import { ExtensionFlavor } from '../../services/generate/extension.js';
import { schema } from '@shopify/cli-kit/node/schema';
export type FunctionConfigType = schema.infer<typeof BaseFunctionConfigurationSchema>;
/**
 * Specification with all the needed properties and methods to load a function.
 */
export interface FunctionSpec<TConfiguration extends FunctionConfigType = FunctionConfigType> extends GenericSpecification {
    identifier: string;
    externalIdentifier: string;
    externalName: string;
    helpURL?: string;
    gated: boolean;
    templateURL: string;
    supportedFlavors: {
        name: string;
        value: ExtensionFlavor;
    }[];
    configSchema: ZodSchemaType<TConfiguration>;
    registrationLimit: number;
    templatePath: (lang: string) => string;
    category: () => ExtensionCategory;
}
/**
 * Class that represents an instance of a local function
 * Before creating this class we've validated that:
 * - There is a spec for this type of function
 * - The Config Schema for that spec is followed by the function config toml file
 *
 * This class holds the public interface to interact with functions
 */
export declare class FunctionInstance<TConfiguration extends FunctionConfigType = FunctionConfigType> implements FunctionExtension {
    idEnvironmentVariableName: string;
    localIdentifier: string;
    directory: string;
    entrySourceFilePath?: string;
    configuration: TConfiguration;
    configurationPath: string;
    private specification;
    constructor(options: {
        configuration: TConfiguration;
        configurationPath: string;
        specification: FunctionSpec<TConfiguration>;
        directory: string;
        entryPath?: string;
    });
    get graphQLType(): string;
    get identifier(): string;
    get type(): string;
    get externalType(): string;
    get name(): string;
    get buildCommand(): string | undefined;
    get inputQueryPath(): string;
    get buildWasmPath(): string;
    get isJavaScript(): boolean;
    publishURL(options: {
        orgId: string;
        appId: string;
    }): Promise<string>;
}
/**
 * Partial FunctionSpec type used when creating a new FunctionSpec, the only mandatory fields are the identifier and the templatePath
 */
export interface CreateFunctionSpecType<TConfiguration extends FunctionConfigType = FunctionConfigType> extends Partial<FunctionSpec<TConfiguration>> {
    identifier: string;
    templatePath: (lang: string) => string;
}
/**
 * Create a new function spec.
 *
 * Everything but "identifer" and "templatePath" is optional.
 * ```ts
 * identifier: string // unique identifier for the function type
 * externalIdentifier: string // unique identifier used externally (default: same as "identifier")
 * externalName: string // human name used externally (default: same as "identifier")
 * helpURL?: string // URL to documentation
 * gated: boolean // whether the function is only accessible to shopifolk or not (default: false)
 * supportedFlavors: {name: string; value: string}[] // list of supported flavors (default: 'wasm' and 'rust')
 * configSchema: ZodSchemaType<TConfiguration> // schema for the function toml file (default: BaseFunctionConfigurationSchema)
 * registrationLimit: number // max number of functions of this type that can be registered (default: 10)
 * templateURL?: string // URL to the functions repository (default: 'https://github.com/Shopify/function-examples')
 * templatePath: (lang: string) => string // path to the template directory for the given language inside the templateURL repo
 * ```
 */
export declare function createFunctionSpecification<TConfiguration extends FunctionConfigType = FunctionConfigType>(spec: CreateFunctionSpecType<TConfiguration>): FunctionSpec;
