import type { ClientApplication } from '../client/types';
import type { ActionSet } from './ActionSet';
import type { ErrorAction } from './Error';
/**
 * Various action groups.
 * @public
 */
export declare enum Group {
    AuthCode = "AuthCode",
    Button = "Button",
    ButtonGroup = "ButtonGroup",
    Cart = "Cart",
    Client = "Client",
    ContextualSaveBar = "ContextualSaveBar",
    Error = "Error",
    Features = "Features",
    FeedbackModal = "FeedbackModal",
    Fullscreen = "Fullscreen",
    LeaveConfirmation = "LeaveConfirmation",
    Link = "Link",
    Loading = "Loading",
    Menu = "Menu",
    Modal = "Modal",
    Navigation = "Navigation",
    Performance = "Performance",
    Pos = "Pos",
    Print = "Print",
    ResourcePicker = "Resource_Picker",
    /**
     * @unstable This API may be updated without warning in the future
     */
    unstable_Picker = "unstable_Picker",
    Scanner = "Scanner",
    SessionToken = "SessionToken",
    Share = "Share",
    TitleBar = "TitleBar",
    Toast = "Toast",
    MarketingExternalActivityTopBar = "MarketingExternalActivityTopBar",
    WebVitals = "WebVitals"
}
/**
 * @internal
 */
export declare enum ComponentType {
    Button = "Button",
    ButtonGroup = "ButtonGroup"
}
/**
 * Base action interface.
 * @remarks
 * All action implementations should inherit from this interface.
 * @internalremarks
 * Should we remove the extraProps definition here, pushing it on sub-types?
 * @public
 */
export interface AnyAction {
    type: any;
    [extraProps: string]: any;
}
export interface ClientInterface {
    name?: string;
    version?: string;
}
/**
 * @public
 */
export interface MetaAction extends AnyAction {
    clientInterface?: ClientInterface;
    readonly version: string;
    readonly group: string;
    readonly type: string;
    payload?: any;
}
/**
 * @public
 */
export interface ClickAction extends MetaAction {
    payload: {
        id: string;
        payload?: any;
    };
}
/**
 * @public
 */
export interface ActionCallback {
    (data: any): void;
}
/**
 * @public
 */
export interface ErrorCallback {
    (data: ErrorAction): void;
}
/**
 * @public
 */
export interface UpdateSubscribe {
    (group: string, subgroups: string[]): void;
}
/**
 * @public
 */
export interface Unsubscribe {
    (): void;
}
/**
 * @public
 */
export interface ErrorSubscriber {
    (callback: ErrorCallback, id?: string): Unsubscribe;
}
/**
 * @internal
 */
export interface ActionSubscription {
    component: Component;
    eventType: string;
    callback: ActionCallback;
    unsubscribe: Unsubscribe;
    updateSubscribe: UpdateSubscribe;
}
/**
 * @internal
 */
export interface UpdateSubscription {
    (subscriptionToRemove: ActionSubscription, group: string, subgroups: string[]): void;
}
/**
 * @public
 */
export interface Component {
    readonly id: string;
    readonly type: string;
    subgroups?: string[];
}
/**
 * @public
 */
export interface ActionSetInterface extends Component {
    readonly app: ClientApplication;
    readonly defaultGroup: string;
    group: string;
    component: Component;
    subscriptions: ActionSubscription[];
    updateSubscription: UpdateSubscription;
    error: ErrorSubscriber;
    subscribe(eventName: string, callback: ActionCallback, component?: Component, currentIndex?: number): Unsubscribe;
    unsubscribe(resetOnly: boolean): ActionSetInterface;
}
/**
 * @public
 */
export interface DispatchAction {
    type: string;
    payload: any;
}
/**
 * @public
 */
export interface SimpleDispatch {
    dispatch(action: string): ActionSet;
}
/**
 * @public
 */
export interface ComplexDispatch<P> {
    dispatch(action: string, payload: P): ActionSet;
}
/**
 * @public
 */
export interface ActionSetProps<T, P> extends SimpleDispatch {
    options: T;
    payload: P;
    set(options: Partial<T>): ActionSet;
}
/**
 * @public
 */
export interface ActionSetPayload<P> extends SimpleDispatch {
    payload: P;
}
/**
 * @public
 */
export interface ActionSetOptions<T> {
    options: T;
    set(options: Partial<T>): ActionSet;
}
/**
 * @public
 */
export interface Dispatch<_> {
    <A extends AnyAction>(action: A): A;
}
