import { loadThemeSpecifications, loadUIExtensionSpecifications, loadFunctionSpecifications, } from '../../models/extensions/specifications.js';
import { ExtensionSpecificationsQuery, } from '../../api/graphql/extension_specifications.js';
import { getArrayRejectingUndefined } from '@shopify/cli-kit/common/array';
import { partnersRequest } from '@shopify/cli-kit/node/api/partners';
/**
 * Returns all extension/function specifications the user has access to.
 * This includes:
 * - UI extensions
 * - Theme extensions
 * - Functions
 *
 * Will return a merge of the local and remote specifications (remote values override local ones)
 * Will only return the specifications that are also defined locally
 * (Functions are not validated againts remote specifications, gated access is defined locally)
 *
 * @param token - Token to access partners API
 * @returns List of extension specifications
 */
export async function fetchSpecifications({ token, apiKey, config, }) {
    const result = await partnersRequest(ExtensionSpecificationsQuery, token, {
        api_key: apiKey,
    });
    const extensionSpecifications = result.extensionSpecifications
        .filter((specification) => specification.options.managementExperience === 'cli')
        .map((spec) => {
        const newSpec = spec;
        // WORKAROUND: The identifiers in the API are different for these extensions to the ones the CLI
        // has been using so far. This is a workaround to keep the CLI working until the API is updated.
        if (spec.identifier === 'theme_app_extension')
            spec.identifier = 'theme';
        if (spec.identifier === 'subscription_management')
            spec.identifier = 'product_subscription';
        newSpec.registrationLimit = spec.options.registrationLimit;
        newSpec.surface = spec.features?.argo?.surface;
        // Hardcoded value for the post purchase extension because the value is wrong in the API
        if (spec.identifier === 'checkout_post_purchase')
            newSpec.surface = 'post_purchase';
        return newSpec;
    });
    const ui = await loadUIExtensionSpecifications(config);
    const theme = await loadThemeSpecifications();
    const functions = await loadFunctionSpecifications(config);
    const local = [...ui, ...theme];
    const updatedSpecs = mergeLocalAndRemoteSpecs(local, extensionSpecifications);
    return [...updatedSpecs, ...functions];
}
function mergeLocalAndRemoteSpecs(local, remote) {
    const updated = local.map((spec) => {
        const remoteSpec = remote.find((remote) => remote.identifier === spec.identifier);
        if (remoteSpec)
            return { ...spec, ...remoteSpec };
        return undefined;
    });
    return getArrayRejectingUndefined(updated);
}
//# sourceMappingURL=fetch-extension-specifications.js.map