import { bundleExtension } from '../../extensions/bundle.js';
import { AbortController } from '@shopify/cli-kit/node/abort';
import { joinPath } from '@shopify/cli-kit/node/path';
import { outputDebug } from '@shopify/cli-kit/node/output';
export async function setupBundlerAndFileWatcher(options) {
    const { default: chokidar } = await import('chokidar');
    const abortController = new AbortController();
    const bundlers = [];
    // eslint-disable-next-line @typescript-eslint/no-misused-promises
    options.devOptions.extensions.forEach(async (extension) => {
        bundlers.push(bundleExtension({
            minify: false,
            outputBundlePath: extension.outputBundlePath,
            environment: 'development',
            env: {
                ...(options.devOptions.app.dotenv?.variables ?? {}),
                APP_URL: options.devOptions.url,
            },
            stdin: {
                contents: extension.getBundleExtensionStdinContent(),
                resolveDir: extension.directory,
                loader: 'tsx',
            },
            stderr: options.devOptions.stderr,
            stdout: options.devOptions.stdout,
            watchSignal: abortController.signal,
            watch: (error) => {
                outputDebug(`The Javascript bundle of the UI extension with ID ${extension.devUUID} has ${error ? 'an error' : 'changed'}`, error ? options.devOptions.stderr : options.devOptions.stdout);
                options.payloadStore
                    .updateExtension(extension, options.devOptions, {
                    status: error ? 'error' : 'success',
                })
                    // ESBuild handles error output
                    .then((_) => { })
                    .catch((_) => { });
            },
        }));
        const localeWatcher = chokidar
            .watch(joinPath(extension.directory, 'locales', '**.json'))
            .on('change', (event, path) => {
            outputDebug(`Locale file at path ${path} changed`, options.devOptions.stdout);
            options.payloadStore
                .updateExtension(extension, options.devOptions)
                .then((_closed) => {
                outputDebug(`Notified extension ${extension.devUUID} about the locale change.`, options.devOptions.stdout);
            })
                // eslint-disable-next-line @typescript-eslint/no-explicit-any
                .catch((_) => { });
        });
        abortController.signal.addEventListener('abort', () => {
            outputDebug(`Closing locale file watching for extension with ID ${extension.devUUID}`, options.devOptions.stdout);
            localeWatcher
                .close()
                .then(() => {
                outputDebug(`Locale file watching closed for extension with ${extension.devUUID}`, options.devOptions.stdout);
            })
                // eslint-disable-next-line @typescript-eslint/no-explicit-any
                .catch((error) => {
                outputDebug(`Locale file watching failed to close for extension with ${extension.devUUID}: ${error.message}`, options.devOptions.stderr);
            });
        });
    });
    await Promise.all(bundlers);
    return {
        close: () => {
            abortController.abort();
        },
    };
}
//# sourceMappingURL=bundler.js.map